
$PROGRAM_NAME = "PIMS Installation script"
$VERSION = "1.0.3"
$COPYRIGHT = "(c) 2019 Planer Limited"
$USER_ACCOUNT_NAME = "__Planer.PIMSUser__"
$USER_ACCOUNT_FULL_NAME = "PIMS User"

$global:INSTALLER_NAME = "CT37_BT37PIMS_windows-x64_1_12_270.exe"
$global:ApplicationPath = Join-Path -Path "$env:ProgramFiles" -ChildPath "CT37_BT37PIMS\CT37_BT37PIMS.exe"

$global:WarningCount = 0
$global:ScriptPath = Split-Path -parent $PSCommandPath
$global:RestrictedFiles = "RestrictedApps.txt"

  
function ElevateScript() {
    If (-NOT ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator))
    {
        try {
          Info "Re-launching script with elevated permissions." 
          $CommandLine = '-ExecutionPolicy RemoteSigned -File "' + $PSCommandPath + '"'
          Start-Process -FilePath powershell.exe -WorkingDirectory "$global:ScriptPath" -Verb RunAs -ArgumentList $CommandLine
      } catch {
        Warning "Unable to run with administrators settings. The installation will end."
        pause
      }
      exit
    } else {
        Info "Running in ", $global:ScriptPath
    }
}


function PadLine() {
  param($message)
  return $message.PadRight(120, ' ') 
}

function UncountedWarning() {
    param($msg)
    foreach ($s in $msg) {
        $padded = PadLine($s)
        Write-Host -ForegroundColor white -BackgroundColor red $padded
    }
    Write-Host ""
}

function Warning() {
    param($msg)
    $global:WarningCount++
    foreach ($s in $msg) {
        $padded = PadLine($s)
        Write-Host -ForegroundColor white -BackgroundColor red $padded
    }
    Write-Host ""
}

function Info() {
    param($msg)
    foreach ($s in $msg) {
        $padded = PadLine($s)
        Write-Host $padded
    }
    Write-Host ""
}


function Title() {
    param($msg)
    $blankline = PadLine " "
    Write-Host -ForegroundColor black -BackgroundColor yellow $blankline
    foreach ($s in $msg) {
        $padded = PadLine($s)
        Write-Host -ForegroundColor black -BackgroundColor yellow $padded
    }
    Write-Host -ForegroundColor black -BackgroundColor yellow $blankline
    
    Write-Host ""
}

function Heading() {
    param($msg)
    Write-Host ""
    foreach ($s in $msg) {
        $padded = PadLine($msg)
        Write-Host -ForegroundColor black -BackgroundColor white $padded
    }
    Write-Host ""
}


function InstallPIMS() {
    Info "Waiting for completion of PIMS installation."
    try {
        $installer = Join-Path -Path $global:ScriptPath -ChildPath $global:INSTALLER_NAME
        $ex = Start-Process "$installer" -Wait -PassThru -WorkingDirectory "$global:ScriptPath" -ArgumentList '-Vfull-kiosk-mode=TRUE'
        return ($ex.ExitCode -eq 0)
    } catch {
        Warning "Unable to install the main application $global:INSTALLER_NAME."
    }
    return $false
}

function DoesUserExist() {
    param($name)
    $pimsuser=Get-LocalUser -Name $name -ErrorAction SilentlyContinue
    
    if ($pimsuser.Enabled -eq "True") {
        return $true
    } else {
        return $false
    }
}

function AskToCreateNewUser() {
    $title = "Create new user."
    $msg = "Do you want to create a user who will be logged in automatically when the PC starts?"
    $yes = New-Object Management.Automation.Host.ChoiceDescription '&Yes'
    $no = New-Object Management.Automation.Host.ChoiceDescription '&No'
    $options = [Management.Automation.Host.ChoiceDescription[]]($yes, $no)
    $default = 0 # &yes
    $response = $Host.UI.PromptForChoice($title, $msg, $options, $default)

    if ($response -eq 0) {
        return $true
    } else {
        return $false
    }

}

function CreatePassword() {
    param($msg)
    
    while ($true) {
        while ($true) {
            Info $msg
            $Password1 = Read-Host -AsSecureString
            $Password1_text = [Runtime.InteropServices.Marshal]::PtrToStringAuto([Runtime.InteropServices.Marshal]::SecureStringToBSTR($Password1))
            if($Password1_text -eq "") {
                Info "Password cannot be empty."
            } else {
                break;
            }
        }
        Info "Re-enter the password."
        $Password2 = Read-Host -AsSecureString
        
        $Password2_text = [Runtime.InteropServices.Marshal]::PtrToStringAuto([Runtime.InteropServices.Marshal]::SecureStringToBSTR($Password2))
        
        if ($Password1_text -eq $Password2_text) {
            return $Password1
        }
        Info "The passwords do not match"

    }
}

function YesOrNo() {
    param($msg)
    $title = "Question"
    $yes = New-Object Management.Automation.Host.ChoiceDescription '&Yes'
    $no = New-Object Management.Automation.Host.ChoiceDescription '&No'
    $options = [Management.Automation.Host.ChoiceDescription[]]($yes, $no)
    $default = 0 # &yes
    $response = $Host.UI.PromptForChoice($title, $msg, $options, $default)

    if ($response -eq 0) {
        return $true
    } else {
        return $false
    }

}

function CreateNewUser() {
    $trying = $true
    while ($trying) {
        try {
            $Password = CreatePassword "Please enter the password to use for the new user"
            $User = New-LocalUser -Name $USER_ACCOUNT_NAME  -FullName "$USER_ACCOUNT_FULL_NAME" -Password $Password -PasswordNeverExpires -UserMayNotChangePassword -Description "User specifically for running PIMS"
        } catch {
            Info $_.Exception.GetType().FullName, $_.Exception.Message
        }
        if (-NOT($User)) {
            Info "Failed to create user";
            $trying = YesOrNo("Do you want to try again?")
        } else {
            $trying = $false;
        }
    }

    if ($User) {
        Info "Created a new user $USER_ACCOUNT_NAME"
        SetUserToAutoLogin $USER_ACCOUNT_NAME $Password
        ShowLocalUsersOnLogin
    } else {
        Warning "No user created so autologin will not work.",
        "If the PC restarts the user will not be automatically logged in and monitoring will be stopped."
    }

    return $User
}

function SetUserToAutoLogin() {
    param($username, $password)
    

    $keyroot = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon"
    $plainPassword = [Runtime.InteropServices.Marshal]::PtrToStringAuto([Runtime.InteropServices.Marshal]::SecureStringToBSTR($password))

    Set-ItemProperty -Path "$keyroot" -Name "DefaultDomainName" -Value $env:computername
    Set-ItemProperty -Path "$keyroot" -Name "DefaultUserName" -Value $username
    Set-ItemProperty -Path "$keyroot" -Name "DefaultPassword" -Value $plainPassword
    Set-ItemProperty -Path "$keyroot" -Name "AutoAdminLogon" -Value "1"

}

function ShowLocalUsersOnLogin() {
    Info "The user should automatically log in but sometimes this may need to be done manually.",
    "To help users remember the account name, the local accounts can be displayed at the login screen"
    $display = YesOrNo "Do you want to display local accounts?"
    If ($display -eq $true) {
        Set-ItemProperty -Path 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Authentication\LogonUI\UserSwitch' -Name Enabled -Value 1
    }
}

function AskToChangePowerSettings() {
    Info "To ensure undisrupted monitoring, it is recommended that you adjust the PC's power settings so that the PC does",
    "not go to sleep."
    $title = "Change power settings."
    $msg = "Do you want the power settings to be optimised now?"
    $yes = New-Object Management.Automation.Host.ChoiceDescription '&Yes'
    $no = New-Object Management.Automation.Host.ChoiceDescription '&No'
    $options = [Management.Automation.Host.ChoiceDescription[]]($yes, $no)
    $default = 0 # &yes
    $response = $Host.UI.PromptForChoice($title, $msg, $options, $default)

    if ($response -eq 0) {
        return $true
    } else {
        return $false
    }

}

function ChangePowerSettings() {
    Info "Optimising the power settings."
    try {
        Info "Changing the standby timeout when powered from the mains to NEVER."
        Start-Process powercfg.exe -Wait -ArgumentList " /change standby-timeout-ac 0"
        Info "Changing the hibernate timeout when powered from the mains to NEVER."
        Start-Process powercfg.exe -Wait -ArgumentList " /change hibernate-timeout-ac 0"
        Info "Changing the monitor timeout when powered from the mains to NEVER."
        Start-Process powercfg.exe -Wait -ArgumentList " /change monitor-timeout-ac 0"
        Info "Changing the hard disk timeout when powered from the mains to NEVER."
        Start-Process powercfg.exe -Wait -ArgumentList " /change disk-timeout-ac 0"
    } catch {
        Warning "Unable to optimise power settings."
    }
}

function DisplayLicence() {
    Info "Press enter to display the licence agreement which will open in a new browser window.",
    "You can close the browser window after you have read the licence."
    pause

    try {
        Start-Process AboutAndLicence.html -WorkingDirectory "$global:ScriptPath" -Wait
    } catch {
        Warning "The licence agreement cannot be found.",
        "Installation will not continue."
        pause
        exit
    }
}

function AcceptLicence() {
    $title = "Licence agreement."
    $msg = "Do you accept the terms of the licence agreement displayed in the browser window?"
    $yes = New-Object Management.Automation.Host.ChoiceDescription '&Yes'
    $no = New-Object Management.Automation.Host.ChoiceDescription '&No'
    $options = [Management.Automation.Host.ChoiceDescription[]]($yes, $no)
    $default = 1 # &no
    $response = $Host.UI.PromptForChoice($title, $msg, $options, $default)
    if ($response -eq 0) {
        return $true
    } else {
        return $false
    }

}

function DisplayWelcomeLockdown() {
    Info "This script will lockdown the PC so that $USER_ACCOUNT_FULL_NAME has limited",
    "access to the PC. Only PIMS will be permitted to be run."
}

function DisplayWelcome() {
    Info "This script will install the Planer Incubator Monitoring Software (PIMS).",
    "",
    "The script is primarily for use on PCs where PIMS will be the main application and where the application should start",
    "automatically. During the installation, you will be able to configure the PC for optimum performance with PIMS.",
    "Part of this optimisation will allow you to configure the PC so that PIMS starts automatically without",
    "requiring a user to log in.",
    "",
    "If you do not want PIMS to run automatically, it is recommended that you end this script and use the alternative",
    "CT37_BT37PIMS_windows-x64_*_*_*.exe installer. This alternative installer will provide you with more flexibility",
    "but will not configure the PC for automatic operation." 
}

function AskToContinue() {
    $title = "Installation."
    $msg = "Is this the correct installation for you?"
    $yes = New-Object Management.Automation.Host.ChoiceDescription '&Yes'
    $no = New-Object Management.Automation.Host.ChoiceDescription '&No'
    $options = [Management.Automation.Host.ChoiceDescription[]]($yes, $no)
    $default = 1 # &no
    $response = $Host.UI.PromptForChoice($title, $msg, $options, $default)
    if ($response -eq 0) {
        return $true
    } else {
        return $false
    }

}

function CreateNewKey() {
    param($root, $keyname)
    $newkey = "$root\$keyname"
    if (-NOT(Test-Path $newkey)) {
        $result = New-Item -Path "$newkey"
        # close handle so we will be able to unload the users repository
        $result.Handle.Close()
    }
    return $newkey
}

function SetProperty() {
    param($key, $name, $value)
    Set-ItemProperty -Path "$key" -Name $name -Value $value
}


function GetUserProfilePath() {
    param($username)
    $sid = (New-Object System.Security.Principal.NTAccount($username)).Translate([System.Security.Principal.SecurityIdentifier]).value
    $pimsuser = gwmi win32_userprofile | Select-Object SID,LocalPath | Where-Object -Property SID -eq -Value $sid
    return $pimsuser.LocalPath
}


function PreventEdgeSwipe() {
    param($registryRoot)
    Info "Preventing edge swipes for registry entry $registryRoot"
    $edgeKey = CreateNewKey "$registryRoot\SOFTWARE\Policies\Microsoft\Windows" "EdgeUI"
    SetProperty $edgeKey "AllowEdgeSwipe" 0
    $edgeKey = CreateNewKey "HKLM:\SOFTWARE\Policies\Microsoft\Windows" "EdgeUI"
    SetProperty $edgeKey "AllowEdgeSwipe" 0
}

function IsUserAccountReadyForMods() {
    $key ="$registryRoot\Software\Microsoft\Windows\CurrentVersion\Explorer\StuckRects3"
    if (-Not(Test-Path "$key")) {
        return $false
    }
    return $true
}

function SetShell() {
    param($registryRoot)
    Info "Setting shell for registry entry $registryRoot"
    
    SetProperty "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\IniFileMapping\system.ini\boot" "Shell" "USR:Software\Microsoft\Windows NT\CurrentVersion\Winlogon"
    # (No need to change default) SetProperty "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\Winlogon" "Shell" "explorer.exe"
    $userKey = CreateNewKey "$registryRoot\Software\Microsoft\Windows NT\CurrentVersion" "Winlogon"
    SetProperty "$userKey" "Shell" "$global:ApplicationPath"

}

function HideDesktop() {
    param($registryRoot)
    Info "Hiding desktop for registry entry $registryRoot"
    $key ="$registryRoot\Software\Microsoft\Windows\CurrentVersion\Policies\Explorer"
    SetProperty $key "NoDesktop" 1
}

function HideTaskBar() {
    param($registryRoot)
    Info "Hiding taskbar for registry entry $registryRoot"
    $key ="$registryRoot\Software\Microsoft\Windows\CurrentVersion\Explorer\StuckRects3"
    
    $settings = Get-ItemProperty -Path "$key"
    $value = $settings.Settings
    $value[8] = 3
    Set-ItemProperty -Path $key -Name "Settings" -Value $value
}

function CreateFolder() {
    param($parent, $name)
    $folder = Join-Path -Path "$parent" -ChildPath "$name"
    if ( -Not(Test-Path -Path "$folder")) {
        $folder = New-Item -Path "$parent" -Name "$name" -ItemType "directory"
    }
    return $folder
}

function SetWallpaper() {
    param($registryRoot)
    Info "Configuring desktop background for registry entry $registryRoot"
    $targetRoot = Join-Path -Path $env:ProgramData -ChildPath "Planer Limited"
    $targetCommon = Join-Path "$targetRoot" -ChildPath "Common"
    $destFolder = CreateFolder "$targetCommon" "Images"
 
    $desktopImageSrc = Join-Path -Path "$global:ScriptPath" -ChildPath "wallpaper.png"
    $desktopImageDest = Join-Path -Path "$destFolder" -ChildPath "wallpaper.png"
    
    if (-Not(Test-Path -Path "$desktopImageDest" -PathType Leaf)) {
        Info "Installing common files."
        Copy-Item "$desktopImageSrc" -Destination "$desktopImageDest" 
    }
    Info "Setting wallpaper"
    $key = "$registryRoot\Control Panel\Desktop"
    SetProperty $key "wallpaper" "$desktopImageDest"
    Info "Configuring lock screen image."
    $personalKey = CreateNewKey "HKLM:\SOFTWARE\Policies\Microsoft\Windows" "Personalization"
    SetProperty "$personalKey" "LockScreenImage" $desktopImageDest
}



function ConfigureUserAccount() {
    $RegistryRoot = "Registry::\HKEY_USERS\$USER_ACCOUNT_NAME"
    $alreadyLoaded = Test-Path $RegistryRoot
    if ($alreadyLoaded) {
        Warning "A registry entry already exists for $RegistryRoot", "This PC cannot be locked down."
    }

    $userspath = GetUserProfilePath($USER_ACCOUNT_NAME)
    Info "Loading the $USER_ACCOUNT_NAME registry settings from $userspath."

    reg load HKU\$USER_ACCOUNT_NAME "$userspath\NTUSER.dat"

    if (IsUserAccountReadyForMods) {
        # the following options should reactivated if not setting the shell. If the shell is set, there is no need to set these.
        # PreventEdgeSwipe $RegistryRoot
        # HideTaskBar $RegistryRoot
        # SetWallpaper $RegistryRoot
        # HideDesktop $RegistryRoot
        
        SetShell $RegistryRoot
    

        $policiesKey = "$RegistryRoot\SOFTWARE\Microsoft\windows\CurrentVersion\Policies"

        $explorerKey = CreateNewKey $policiesKey "Explorer"

        SetProperty $explorerKey "DisallowRun" "1"
        SetProperty $explorerKey "RestrictRun" "1"

        $restrictKey = CreateNewKey $explorerKey "RestrictRun"

        SetProperty $restrictKey "1" "downloader.exe"
        SetProperty $restrictKey "2" "CT37_BT37PIMS.exe"
        SetProperty $restrictKey "3" "CT37_BT37PIMS_UPDATE.exe"
        SetProperty $restrictKey "4" "CT37_BT37PIMS.lnk"
        SetProperty $restrictKey "5" "MicrosoftEdge.exe"
        SetProperty $restrictKey "6" "iexplore.exe"
        SetProperty $restrictKey "7" "explorer.exe"
        SetProperty $restrictKey "8" "java.exe"
        SetProperty $restrictKey "9" "javaw.exe"
        Info "Applications that can be run by $USER_ACCOUNT_NAME now restricted."
    } else {
        Warning "Have you logged in as $USER_ACCOUNT_NAME and then logged out again?",
           "It is important that you log in once as $USER_ACCOUNT_NAME and then log out",
           "again. Only then you should log in as an administrator to re-run this script."
    }

    
    Info "Unloading the $USER_ACCOUNT_NAME registry settings."
    [gc]::Collect()
    reg unload HKU\$USER_ACCOUNT_NAME
}



function InitialInstallation()  {
    DisplayWelcome
    if (-NOT(YesOrNo "Is this this correct installation for you?")) {
        Warning "The application will not be installed."
        pause 
        exit
    }

    Heading "Application installation"
    if (-Not(InstallPIMS)) {
        Warning "Installation cancelled."
        pause
        exit
    }


    Heading "User configuration"

    if (DoesUserExist($USER_ACCOUNT_NAME) -eq $true) {
        Warning "The user account $USER_ACCOUNT_NAME already exists and so will not be recreated."
    } elseif (AskToCreateNewUser -eq $true) {
        $User = CreateNewUser($USER_ACCOUNT_NAME)
    } else {
        Warning "You chose not to create new user.",
        "If the PC restarts the user will not be automatically logged in and monitoring will be stopped."
    }


    Heading "Power settings"
    if (AskToChangePowerSettings -eq $true) {
        ChangePowerSettings
    } else {
        Warning "You have selected not to change the power settings.",
        "If the PC goes to sleep, monitoring will be disrupted."
    }



    Heading "End of installation."
    Info "The installation is now complete."

    if ($User) {
        Info "After installation, you should restart the PC to check that the user is automatically logged in.", 
        "If the user is not logged in automatically, it is possible that Exchange Active Sync (EAS) password restrictions",
        "are active."
        "In this case, you will need to ask your system administrator to configure the system for automatic log in.",
        "",
        "After you have logged in as $USER_ACCOUNT_FULL_NAME, you should log out and then log back into this",
        "account to re-run this script and lock down the PC to restrict the user's access."
    }
}

function LockdownExistingUser() {
    DisplayWelcomeLockdown
    if (-NOT(YesOrNo "Do you want to lock down this PC for $USER_ACCOUNT_FULL_NAME")) {
        Warning "The PC has not been locked down."
        pause 
        exit
    }
    Heading "Restrict access"
    ConfigureUserAccount
}




# START OF INSTALLATION
ElevateScript

Title $PROGRAM_NAME, $VERSION, $COPYRIGHT
if (DoesUserExist($USER_ACCOUNT_NAME) -eq $true) {
    LockdownExistingUser
} else {
    InitialInstallation
}

if ($global:WarningCount -gt 0) {
    Warning "$global:WarningCount warnings occurred. Please review the results above."
}
pause

# SIG # Begin signature block
# MIIuzwYJKoZIhvcNAQcCoIIuwDCCLrwCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCD72O/aTUaiXc4q
# 0Wyg4YgtSX9ij9AaQeve67/3yxG/ZaCCFEMwggWiMIIEiqADAgECAhB4AxhCRXCK
# Qc9vAbjutKlUMA0GCSqGSIb3DQEBDAUAMEwxIDAeBgNVBAsTF0dsb2JhbFNpZ24g
# Um9vdCBDQSAtIFIzMRMwEQYDVQQKEwpHbG9iYWxTaWduMRMwEQYDVQQDEwpHbG9i
# YWxTaWduMB4XDTIwMDcyODAwMDAwMFoXDTI5MDMxODAwMDAwMFowUzELMAkGA1UE
# BhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNpZ24gbnYtc2ExKTAnBgNVBAMTIEdsb2Jh
# bFNpZ24gQ29kZSBTaWduaW5nIFJvb3QgUjQ1MIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAti3FMN166KuQPQNysDpLmRZhsuX/pWcdNxzlfuyTg6qE9aND
# m5hFirhjV12bAIgEJen4aJJLgthLyUoD86h/ao+KYSe9oUTQ/fU/IsKjT5GNswWy
# KIKRXftZiAULlwbCmPgspzMk7lA6QczwoLB7HU3SqFg4lunf+RuRu4sQLNLHQx2i
# CXShgK975jMKDFlrjrz0q1qXe3+uVfuE8ID+hEzX4rq9xHWhb71hEHREspgH4nSr
# /2jcbCY+6R/l4ASHrTDTDI0DfFW4FnBcJHggJetnZ4iruk40mGtwEd44ytS+ocCc
# 4d8eAgHYO+FnQ4S2z/x0ty+Eo7+6CTc9Z2yxRVwZYatBg/WsHet3DUZHc86/vZWV
# 7Z0riBD++ljop1fhs8+oWukHJZsSxJ6Acj2T3IyU3ztE5iaA/NLDA/CMDNJF1i7n
# j5ie5gTuQm5nfkIWcWLnBPlgxmShtpyBIU4rxm1olIbGmXRzZzF6kfLUjHlufKa7
# fkZvTcWFEivPmiJECKiFN84HYVcGFxIkwMQxc6GYNVdHfhA6RdktpFGQmKmgBzfE
# ZRqqHGsWd/enl+w/GTCZbzH76kCy59LE+snQ8FB2dFn6jW0XMr746X4D9OeHdZrU
# SpEshQMTAitCgPKJajbPyEygzp74y42tFqfT3tWbGKfGkjrxgmPxLg4kZN8CAwEA
# AaOCAXcwggFzMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDAzAP
# BgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQfAL9GgAr8eDm3pbRD2VZQu86WOzAf
# BgNVHSMEGDAWgBSP8Et/qC5FJK5NUPpjmove4t0bvDB6BggrBgEFBQcBAQRuMGww
# LQYIKwYBBQUHMAGGIWh0dHA6Ly9vY3NwLmdsb2JhbHNpZ24uY29tL3Jvb3RyMzA7
# BggrBgEFBQcwAoYvaHR0cDovL3NlY3VyZS5nbG9iYWxzaWduLmNvbS9jYWNlcnQv
# cm9vdC1yMy5jcnQwNgYDVR0fBC8wLTAroCmgJ4YlaHR0cDovL2NybC5nbG9iYWxz
# aWduLmNvbS9yb290LXIzLmNybDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAyBggrBgEF
# BQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2lnbi5jb20vcmVwb3NpdG9yeS8wDQYJ
# KoZIhvcNAQEMBQADggEBAKz3zBWLMHmoHQsoiBkJ1xx//oa9e1ozbg1nDnti2eEY
# XLC9E10dI645UHY3qkT9XwEjWYZWTMytvGQTFDCkIKjgP+icctx+89gMI7qoLao8
# 9uyfhzEHZfU5p1GCdeHyL5f20eFlloNk/qEdUfu1JJv10ndpvIUsXPpYd9Gup7EL
# 4tZ3u6m0NEqpbz308w2VXeb5ekWwJRcxLtv3D2jmgx+p9+XUnZiM02FLL8Mofnre
# kw60faAKbZLEtGY/fadY7qz37MMIAas4/AocqcWXsojICQIZ9lyaGvFNbDDUswar
# AGBIDXirzxetkpNiIHd1bL3IMrTcTevZ38GQlim9wX8wggboMIIE0KADAgECAhB3
# vQ4Ft1kLth1HYVMeP3XtMA0GCSqGSIb3DQEBCwUAMFMxCzAJBgNVBAYTAkJFMRkw
# FwYDVQQKExBHbG9iYWxTaWduIG52LXNhMSkwJwYDVQQDEyBHbG9iYWxTaWduIENv
# ZGUgU2lnbmluZyBSb290IFI0NTAeFw0yMDA3MjgwMDAwMDBaFw0zMDA3MjgwMDAw
# MDBaMFwxCzAJBgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTIw
# MAYDVQQDEylHbG9iYWxTaWduIEdDQyBSNDUgRVYgQ29kZVNpZ25pbmcgQ0EgMjAy
# MDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMsg75ceuQEyQ6BbqYoj
# /SBerjgSi8os1P9B2BpV1BlTt/2jF+d6OVzA984Ro/ml7QH6tbqT76+T3PjisxlM
# g7BKRFAEeIQQaqTWlpCOgfh8qy+1o1cz0lh7lA5tD6WRJiqzg09ysYp7ZJLQ8LRV
# X5YLEeWatSyyEc8lG31RK5gfSaNf+BOeNbgDAtqkEy+FSu/EL3AOwdTMMxLsvUCV
# 0xHK5s2zBZzIU+tS13hMUQGSgt4T8weOdLqEgJ/SpBUO6K/r94n233Hw0b6nskEz
# IHXMsdXtHQcZxOsmd/KrbReTSam35sOQnMa47MzJe5pexcUkk2NvfhCLYc+YVaMk
# oog28vmfvpMusgafJsAMAVYS4bKKnw4e3JiLLs/a4ok0ph8moKiueG3soYgVPMLq
# 7rfYrWGlr3A2onmO3A1zwPHkLKuU7FgGOTZI1jta6CLOdA6vLPEV2tG0leis1Ult
# 5a/dm2tjIF2OfjuyQ9hiOpTlzbSYszcZJBJyc6sEsAnchebUIgTvQCodLm3HadNu
# twFsDeCXpxbmJouI9wNEhl9iZ0y1pzeoVdwDNoxuz202JvEOj7A9ccDhMqeC5LYy
# AjIwfLWTyCH9PIjmaWP47nXJi8Kr77o6/elev7YR8b7wPcoyPm593g9+m5XEEofn
# GrhO7izB36Fl6CSDySrC/blTAgMBAAGjggGtMIIBqTAOBgNVHQ8BAf8EBAMCAYYw
# EwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4E
# FgQUJZ3Q/FkJhmPF7POxEztXHAOSNhEwHwYDVR0jBBgwFoAUHwC/RoAK/Hg5t6W0
# Q9lWULvOljswgZMGCCsGAQUFBwEBBIGGMIGDMDkGCCsGAQUFBzABhi1odHRwOi8v
# b2NzcC5nbG9iYWxzaWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUwRgYIKwYBBQUH
# MAKGOmh0dHA6Ly9zZWN1cmUuZ2xvYmFsc2lnbi5jb20vY2FjZXJ0L2NvZGVzaWdu
# aW5ncm9vdHI0NS5jcnQwQQYDVR0fBDowODA2oDSgMoYwaHR0cDovL2NybC5nbG9i
# YWxzaWduLmNvbS9jb2Rlc2lnbmluZ3Jvb3RyNDUuY3JsMFUGA1UdIAROMEwwQQYJ
# KwYBBAGgMgECMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24u
# Y29tL3JlcG9zaXRvcnkvMAcGBWeBDAEDMA0GCSqGSIb3DQEBCwUAA4ICAQAldaAJ
# yTm6t6E5iS8Yn6vW6x1L6JR8DQdomxyd73G2F2prAk+zP4ZFh8xlm0zjWAYCImbV
# YQLFY4/UovG2XiULd5bpzXFAM4gp7O7zom28TbU+BkvJczPKCBQtPUzosLp1pnQt
# pFg6bBNJ+KUVChSWhbFqaDQlQq+WVvQQ+iR98StywRbha+vmqZjHPlr00Bid/XSX
# hndGKj0jfShziq7vKxuav2xTpxSePIdxwF6OyPvTKpIz6ldNXgdeysEYrIEtGiH6
# bs+XYXvfcXo6ymP31TBENzL+u0OF3Lr8psozGSt3bdvLBfB+X3Uuora/Nao2Y8nO
# ZNm9/Lws80lWAMgSK8YnuzevV+/Ezx4pxPTiLc4qYc9X7fUKQOL1GNYe6ZAvytOH
# X5OKSBoRHeU3hZ8uZmKaXoFOlaxVV0PcU4slfjxhD4oLuvU/pteO9wRWXiG7n9dq
# cYC/lt5yA9jYIivzJxZPOOhRQAyuku++PX33gMZMNleElaeEFUgwDlInCI2Oor0i
# xxnJpsoOqHo222q6YV8RJJWk4o5o7hmpSZle0LQ0vdb5QMcQlzFSOTUpEYck08T7
# qWPLd0jV+mL8JOAEek7Q5G7ezp44UCb0IXFl1wkl1MkHAHq4x/N36MXU4lXQ0x72
# f1LiSY25EXIMiEQmM2YBRN/kMw4h3mKJSAfa9TCCB60wggWVoAMCAQICDHhortGl
# 0olZwl9vJzANBgkqhkiG9w0BAQsFADBcMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
# R2xvYmFsU2lnbiBudi1zYTEyMDAGA1UEAxMpR2xvYmFsU2lnbiBHQ0MgUjQ1IEVW
# IENvZGVTaWduaW5nIENBIDIwMjAwHhcNMjQwNDExMTAxNDM0WhcNMjcwNjMwMTEy
# NDI2WjCB8zEdMBsGA1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xETAPBgNVBAUT
# CDAwNjUxNTA0MRMwEQYLKwYBBAGCNzwCAQMTAkdCMQswCQYDVQQGEwJHQjEQMA4G
# A1UECBMHRW5nbGFuZDEaMBgGA1UEBxMRU3VuYnVyeS1vbi1UaGFtZXMxGjAYBgNV
# BAkTETExMCBXaW5kbWlsbCBSb2FkMRcwFQYDVQQKEw5QTEFORVIgTElNSVRFRDEX
# MBUGA1UEAxMOUExBTkVSIExJTUlURUQxITAfBgkqhkiG9w0BCQEWEnN1cHBvcnRA
# cGxhbmVyLmNvbTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMGgA0oU
# cxWiaS2YlQDADXudDKBLVDUHML9c0fqJhfVktpW9hcNM+q78tqPROpampG1JSIeF
# N8aN+XoN98s43EFVlLzOgwRSgZUV7AkMxuNhLcgKGc47qFQxdge/WHrwm7UrdGLv
# wmOKJlqYpyr9zAvBQSrfLG0rdniOQV4hzIlcrPSsexnzLIPuXlCtchnw/4cPQOXn
# HhmFDq3UWldj9W0SvPSQyAPW17GpULaas5bleWKjYfNpVhNK9vHNG3Ptc12BpzQY
# kMbekHTzAczI8N6h8/wxxFbffvPH20ioKLzuTMdgSkueiJ6POh0lZx3YdHnvSFVm
# AuCQvEDucXnGR3XmhSKK/e3RMlj2GvEACqj3+HqdcW+/5qBoZWRqAEktEWnW87lW
# OJ0XwPwCf6m7mv5iAuM3MHMhsvBkXA+aa37VMwvmkgJ+YqDCIwcugokLFN9NJ5ym
# Hi4yz4PppLqVeWen6u3POBx0/mL2zisQQ707T4JU9dGDEZwPI04rVY6NAXTo2DQg
# ADLljbceVxFnowc6/YVetGvq5IintmgZHAlL189yGvft+926zQos5OmaOj9Z4fR6
# m7NUmTTPIgBHIPVwMcnzgST6QakN/mosrUdrQBSEm3qaSmtq73rx97iuACT+ujnv
# fzTRMF3q4otNgR3SUPJojjao4uPrP8sd0dErAgMBAAGjggHVMIIB0TAOBgNVHQ8B
# Af8EBAMCB4AwgZ8GCCsGAQUFBwEBBIGSMIGPMEwGCCsGAQUFBzAChkBodHRwOi8v
# c2VjdXJlLmdsb2JhbHNpZ24uY29tL2NhY2VydC9nc2djY3I0NWV2Y29kZXNpZ25j
# YTIwMjAuY3J0MD8GCCsGAQUFBzABhjNodHRwOi8vb2NzcC5nbG9iYWxzaWduLmNv
# bS9nc2djY3I0NWV2Y29kZXNpZ25jYTIwMjAwVQYDVR0gBE4wTDBBBgkrBgEEAaAy
# AQIwNDAyBggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2lnbi5jb20vcmVw
# b3NpdG9yeS8wBwYFZ4EMAQMwCQYDVR0TBAIwADBHBgNVHR8EQDA+MDygOqA4hjZo
# dHRwOi8vY3JsLmdsb2JhbHNpZ24uY29tL2dzZ2NjcjQ1ZXZjb2Rlc2lnbmNhMjAy
# MC5jcmwwHQYDVR0RBBYwFIESc3VwcG9ydEBwbGFuZXIuY29tMBMGA1UdJQQMMAoG
# CCsGAQUFBwMDMB8GA1UdIwQYMBaAFCWd0PxZCYZjxezzsRM7VxwDkjYRMB0GA1Ud
# DgQWBBThHCCmCINtdGSVmehS/XW7q3nNYjANBgkqhkiG9w0BAQsFAAOCAgEAcvOC
# 0NdJdk0SoSwwm6DKySf8GvUTgQ7qFNuO6g5wZ98WzSUbj95LHUe8Qxw4gfs5qt+F
# frYzcBHYh1EJt633BbslB8M1ZaH5iJd4ub8EDwD5q+hesGj3TlWujQ56rrgxtU7b
# kOCzDx65OKqECfYvxwOal1XvkncTD+bUff+7VduZbBFwjJd6GjnJ5a3PnphVLbTI
# pubjSa/7Rs6MiwWLS8OmhIow04cagFuUlnHlpa68Unll3Sk9XHW3F5MQc22PBzGM
# W+ZW3/pbKbv8BIu7a3iLNZiLWYzrviswmaa4e4x7mrwPrBRKcOCMIraOL+O690kK
# Vk+xn+S3hAnEQd++S1t5nICdOjLZAmY+DFvfxOriV/aONyci39dbvNxXFLf9DMO7
# IhwJVNb3ov6Kgglf5SqAhaIOVrdsMMRY3Pw13CoJza/g5cHcqXzUAaZvEUpaO6Z4
# daP7ZFAqLDN0guLUXPI7vMH97rDeBtJ4hVK7mHUUurKK3d76uGR23V3rZyivjAIw
# gL/7N+2DWTE7CuZtxALyvRXU+ZAzlcnkaVS6JwnKRPzszK++pLelrE6MoVaniREN
# 3sudJJGO1swWMb0HMs364CFmyd+iA0rEbPsLbtcqb6LtWaVz6TnOAJimE3B37ysK
# hCVcUxdMmSK9axZ24fzG1YI4nWGZ/9iyOJE+t60xghniMIIZ3gIBATBsMFwxCzAJ
# BgNVBAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTIwMAYDVQQDEylH
# bG9iYWxTaWduIEdDQyBSNDUgRVYgQ29kZVNpZ25pbmcgQ0EgMjAyMAIMeGiu0aXS
# iVnCX28nMA0GCWCGSAFlAwQCAQUAoHwwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZI
# hvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcC
# ARUwLwYJKoZIhvcNAQkEMSIEILr1Crr1FcFpIta+AtXAquslxsWwlNXTTC4GFixn
# bpDbMA0GCSqGSIb3DQEBAQUABIICADW2rDzvdYcG8u+Pq1mIhfOLmgSYPw7ZKIxr
# Ja0Z0eBZIcWGz+G/CTTBp58SxFTBDQvq94q6k8ve5ZkJ10eIJ1OkuhvSsO0ReZJv
# iLl6xoK3YvGhkGrmvzD1EeVTWFS/xAd1SBdsuYR4nrL7rdW5RMd0f2NQUgRMm/3K
# Kmki2Tls570NowIDvbBDzCjwUJQYg1pfIp+2jD15oy7M7owrlPPoKG9QJa6YULue
# WJDPHWoRPKFukPBGyGTXg74bcuP0dfj2AR8Jcq+3UDzwwcHUxnkyLLxO8GJ9WCnG
# F4tQd/ginPtgKZXvtFphyky/Rws2Xk70ODm8G016IBchzg0QF2kQtWHVWwi+PZfZ
# 5I4C7eF/T+Mzr73yGHCqUILjCvYjPV3QKoiNzVDQXu2m/4BLRizuVNPPMa9YbEEd
# Mo5TW2X7wg8db3i6Ogd61lyIKl8kXVanVriuTqffMnNnorLj/KqXsTM51N2nkZLh
# eEL7UaKMwVEuAmGrsMsdvQeguapa1KTy3htL8EbOJyynMdnQP75zzs3oXbzZgEMY
# EbIsmxOLgFE+ShQX0pLUc2d0wWze2vcUxogJ8uPsT6y93NPv/ESsT76WuhdmCQpJ
# TfWgbC45OVFbdXuuRAKv0Oq8mTW/uriZIT7tE8g+5sCwdgCKoF72HxdPX6V+wBoL
# 92KuwCmKoYIWyTCCFsUGCisGAQQBgjcDAwExgha1MIIWsQYJKoZIhvcNAQcCoIIW
# ojCCFp4CAQMxDTALBglghkgBZQMEAgEwgeUGCyqGSIb3DQEJEAEEoIHVBIHSMIHP
# AgEBBgkrBgEEAaAyAgMwMTANBglghkgBZQMEAgEFAAQga8xuWfrh7pah3OlPZM06
# x1OrMr5yd42LapZeVoX1P2YCFBo3SZEEE/nHv/jtduKw32TCbI8gGA8yMDI0MDgx
# MzA4NDU1NlowAwIBAaBgpF4wXDELMAkGA1UEBhMCQkUxGTAXBgNVBAoMEEdsb2Jh
# bFNpZ24gbnYtc2ExMjAwBgNVBAMMKUdsb2JhbHNpZ24gVFNBIGZvciBBZHZhbmNl
# ZCAtIEc0IC0gMjAyMzExoIISUzCCBmswggRToAMCAQICEAEZdXRxyZLXRN+lluu5
# cBUwDQYJKoZIhvcNAQELBQAwWzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2Jh
# bFNpZ24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGltZXN0YW1waW5nIENB
# IC0gU0hBMzg0IC0gRzQwHhcNMjMxMTAyMTAzMDAyWhcNMzQxMjA0MTAzMDAyWjBc
# MQswCQYDVQQGEwJCRTEZMBcGA1UECgwQR2xvYmFsU2lnbiBudi1zYTEyMDAGA1UE
# AwwpR2xvYmFsc2lnbiBUU0EgZm9yIEFkdmFuY2VkIC0gRzQgLSAyMDIzMTEwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQCyNUZ0qoON1ZanPEjVxcqo31S+
# CKuh31zpSdBgXrWlGvdDWEOXPPRnYwgyPBl/K9lVRtXUjMBcz6TFpRq6pyvOJkIh
# POW7oaOV3WDqElWu787cMoTto7XgP3PRNbibu8VE3eG46/NZrYn2cY9aCvoKkgWE
# DZcBvwW7/FgBs43J1AWFp5ArbqzT2U7apyQ1lm+qs6BBO+D55xGO1WYCgC09zM8e
# pJaLF4DcTDkaJHUsxXcW2ZGDJn/nE4uiRVTmtkp359ItLuewPEjZxo37evQrvKYi
# SKLX3q14R4gMX5v0kUoGHPoDnmpWHisw4/OOWbC0Hx5hOIZ5+YODlI8JMEIztA63
# iIIYLT/XgYsnoGnx0wWuxkWjwh+brenAyE/X58anQTJo/1nKVFz7v9kfFvBS0s+4
# NZWlkc6jHfV2UpjskWGLCaGtmZnorJQolziMCa48nPh+UaI3ashxuh1PDSYBVn5X
# w3VC2FPgY2Pdfp4dqGLozv6ZWVP28wCK/ZOVz9ECAwEAAaOCAagwggGkMA4GA1Ud
# DwEB/wQEAwIHgDAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAdBgNVHQ4EFgQUxL7u
# hzyJdA7es+4ZG4UMzkFOf50wVgYDVR0gBE8wTTAIBgZngQwBBAIwQQYJKwYBBAGg
# MgEeMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3Jl
# cG9zaXRvcnkvMAwGA1UdEwEB/wQCMAAwgZAGCCsGAQUFBwEBBIGDMIGAMDkGCCsG
# AQUFBzABhi1odHRwOi8vb2NzcC5nbG9iYWxzaWduLmNvbS9jYS9nc3RzYWNhc2hh
# Mzg0ZzQwQwYIKwYBBQUHMAKGN2h0dHA6Ly9zZWN1cmUuZ2xvYmFsc2lnbi5jb20v
# Y2FjZXJ0L2dzdHNhY2FzaGEzODRnNC5jcnQwHwYDVR0jBBgwFoAU6hbGaefjy1dF
# OTOk8EC+0MO9ZZYwQQYDVR0fBDowODA2oDSgMoYwaHR0cDovL2NybC5nbG9iYWxz
# aWduLmNvbS9jYS9nc3RzYWNhc2hhMzg0ZzQuY3JsMA0GCSqGSIb3DQEBCwUAA4IC
# AQCzMtHqZ//b36e0N0Rd7R6+diPJzgPtTdRq5zOMPF8gYtvu6Ww4OeWZcfsmkR8n
# sXNcAxnPaDLQ1eZ2eEqqPJcy0hXuehwyPGCnQcq5PvFB6sPT8cflvt4axsGOIt/W
# gOWP8qyyIY14tsSJjJS9MnO42JdEPNdmbA0cEFxeqIhAvaCuTlotZE8GJaWExjhw
# x1RzFI1XFqkwHKgJSd+lAQYDvxOzdJSbB4GvDUGQVSmwYKlU+jggM84Jug5MZ1iB
# hqntiIapmOO25UaXJEdsSNEQaspxsj5dwz0tIYJrg2Nvl8CR/vt9lrmqwBzNpa2Q
# eIDWfW2JKkCOrCX664g2I36G8vu1Bu0ogyyz2pp6b0gRFpQ2tUVAnYE1DcWxjJs7
# 5jzpehhQ+TmKkne7kSJuoLlbKgFAKOTRSKkwjqKGEjdNyVmZx6YDf+GRCn0K+AtC
# DnGu9s+65TH4+R8t8OAKjISMpTmjO7DzNtlD1ZuYJA/QwuMmPq3h+/seq94G9vto
# Qewx36nJHowZ9j72Hpgu0WCBWyZ09FROQATftV7U9+7wDYdvQECnaeooyKGpT3cS
# iTFq6ZqDd4upxUQz7rdpTiy0p7SVeJvWqkAsNhqnREOzUthgxnNXv3zWNdMjo2BC
# ItYWFc4TGunO9eXPWr6sP3Pp+nO/Gc2il2bKHGANor1UzDCCBlkwggRBoAMCAQIC
# DQHsHJJA3v0uQF18R3QwDQYJKoZIhvcNAQEMBQAwTDEgMB4GA1UECxMXR2xvYmFs
# U2lnbiBSb290IENBIC0gUjYxEzARBgNVBAoTCkdsb2JhbFNpZ24xEzARBgNVBAMT
# Ckdsb2JhbFNpZ24wHhcNMTgwNjIwMDAwMDAwWhcNMzQxMjEwMDAwMDAwWjBbMQsw
# CQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMo
# R2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEzODQgLSBHNDCCAiIwDQYJ
# KoZIhvcNAQEBBQADggIPADCCAgoCggIBAPAC4jAj+uAb4Zp0s691g1+pR1LHYTpj
# fDkjeW10/DHkdBIZlvrOJ2JbrgeKJ+5Xo8Q17bM0x6zDDOuAZm3RKErBLLu5cPJy
# roz3mVpddq6/RKh8QSSOj7rFT/82QaunLf14TkOI/pMZF9nuMc+8ijtuasSI8O6X
# 9tzzGKBLmRwOh6cm4YjJoOWZ4p70nEw/XVvstu/SZc9FC1Q9sVRTB4uZbrhUmYqo
# MZI78np9/A5Y34Fq4bBsHmWCKtQhx5T+QpY78Quxf39GmA6HPXpl69FWqS69+1g9
# tYX6U5lNW3TtckuiDYI3GQzQq+pawe8P1Zm5P/RPNfGcD9M3E1LZJTTtlu/4Z+oI
# vo9Jev+QsdT3KRXX+Q1d1odDHnTEcCi0gHu9Kpu7hOEOrG8NubX2bVb+ih0JPiQO
# ZybH/LINoJSwspTMe+Zn/qZYstTYQRLBVf1ukcW7sUwIS57UQgZvGxjVNupkrs79
# 9QXm4mbQDgUhrLERBiMZ5PsFNETqCK6dSWcRi4LlrVqGp2b9MwMB3pkl+XFu6Zxd
# AkxgPM8CjwH9cu6S8acS3kISTeypJuV3AqwOVwwJ0WGeJoj8yLJN22TwRZ+6wT9U
# o9h2ApVsao3KIlz2DATjKfpLsBzTN3SE2R1mqzRzjx59fF6W1j0ZsJfqjFCRba9X
# hn4QNx1rGhTfAgMBAAGjggEpMIIBJTAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/
# BAgwBgEB/wIBADAdBgNVHQ4EFgQU6hbGaefjy1dFOTOk8EC+0MO9ZZYwHwYDVR0j
# BBgwFoAUrmwFo5MT4qLn4tcc1sfwf8hnU6AwPgYIKwYBBQUHAQEEMjAwMC4GCCsG
# AQUFBzABhiJodHRwOi8vb2NzcDIuZ2xvYmFsc2lnbi5jb20vcm9vdHI2MDYGA1Ud
# HwQvMC0wK6ApoCeGJWh0dHA6Ly9jcmwuZ2xvYmFsc2lnbi5jb20vcm9vdC1yNi5j
# cmwwRwYDVR0gBEAwPjA8BgRVHSAAMDQwMgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3
# Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRvcnkvMA0GCSqGSIb3DQEBDAUAA4ICAQB/
# 4ojZV2crQl+BpwkLusS7KBhW1ky/2xsHcMb7CwmtADpgMx85xhZrGUBJJQge5Jv3
# 1qQNjx6W8oaiF95Bv0/hvKvN7sAjjMaF/ksVJPkYROwfwqSs0LLP7MJWZR29f/be
# gsi3n2HTtUZImJcCZ3oWlUrbYsbQswLMNEhFVd3s6UqfXhTtchBxdnDSD5bz6jdX
# lJEYr9yNmTgZWMKpoX6ibhUm6rT5fyrn50hkaS/SmqFy9vckS3RafXKGNbMCVx+L
# nPy7rEze+t5TTIP9ErG2SVVPdZ2sb0rILmq5yojDEjBOsghzn16h1pnO6X1LlizM
# FmsYzeRZN4YJLOJF1rLNboJ1pdqNHrdbL4guPX3x8pEwBZzOe3ygxayvUQbwEccd
# MMVRVmDofJU9IuPVCiRTJ5eA+kiJJyx54jzlmx7jqoSCiT7ASvUh/mIQ7R0w/PbM
# 6kgnfIt1Qn9ry/Ola5UfBFg0ContglDk0Xuoyea+SKorVdmNtyUgDhtRoNRjqoPq
# bHJhSsn6Q8TGV8Wdtjywi7C5HDHvve8U2BRAbCAdwi3oC8aNbYy2ce1SIf4+9p+f
# ORqurNIveiCx9KyqHeItFJ36lmodxjzK89kcv1NNpEdZfJXEQ0H5JeIsEH6B+Q2U
# p33ytQn12GByQFCVINRDRL76oJXnIFm2eMakaqoimzCCBYMwggNroAMCAQICDkXm
# uwODM8OFZUjm/0VRMA0GCSqGSIb3DQEBDAUAMEwxIDAeBgNVBAsTF0dsb2JhbFNp
# Z24gUm9vdCBDQSAtIFI2MRMwEQYDVQQKEwpHbG9iYWxTaWduMRMwEQYDVQQDEwpH
# bG9iYWxTaWduMB4XDTE0MTIxMDAwMDAwMFoXDTM0MTIxMDAwMDAwMFowTDEgMB4G
# A1UECxMXR2xvYmFsU2lnbiBSb290IENBIC0gUjYxEzARBgNVBAoTCkdsb2JhbFNp
# Z24xEzARBgNVBAMTCkdsb2JhbFNpZ24wggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQCVB+hzymb57BTKezz3DQjxtEULLIK0SMbrWzyug7hBkjMUpG9/6SrM
# xrCIa8W2idHGsv8UzlEUIexK3RtaxtaH7k06FQbtZGYLkoDKRN5zlE7zp4l/T3hj
# CMgSUG1CZi9NuXkoTVIaihqAtxmBDn7EirxkTCEcQ2jXPTyKxbJm1ZCatzEGxb7i
# bTIGph75ueuqo7i/voJjUNDwGInf5A959eqiHyrScC5757yTu21T4kh8jBAHOP9m
# sndhfuDqjDyqtKT285VKEgdt/Yyyic/QoGF3yFh0sNQjOvddOsqi250J3l1ELZDx
# gc1Xkvp+vFAEYzTfa5MYvms2sjnkrCQ2t/DvthwTV5O23rL44oW3c6K4NapF8uCd
# NqFvVIrxclZuLojFUUJEFZTuo8U4lptOTloLR/MGNkl3MLxxN+Wm7CEIdfzmYRY/
# d9XZkZeECmzUAk10wBTt/Tn7g/JeFKEEsAvp/u6P4W4LsgizYWYJarEGOmWWWcDw
# Nf3J2iiNGhGHcIEKqJp1HZ46hgUAntuA1iX53AWeJ1lMdjlb6vmlodiDD9H/3zAR
# +YXPM0j1ym1kFCx6WE/TSwhJxZVkGmMOeT31s4zKWK2cQkV5bg6HGVxUsWW2v4yb
# 3BPpDW+4LtxnbsmLEbWEFIoAGXCDeZGXkdQaJ783HjIH2BRjPChMrwIDAQABo2Mw
# YTAOBgNVHQ8BAf8EBAMCAQYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQUrmwF
# o5MT4qLn4tcc1sfwf8hnU6AwHwYDVR0jBBgwFoAUrmwFo5MT4qLn4tcc1sfwf8hn
# U6AwDQYJKoZIhvcNAQEMBQADggIBAIMl7ejR/ZVSzZ7ABKCRaeZc0ITe3K2iT+hH
# eNZlmKlbqDyHfAKK0W63FnPmX8BUmNV0vsHN4hGRrSMYPd3hckSWtJVewHuOmXgW
# QxNWV7Oiszu1d9xAcqyj65s1PrEIIaHnxEM3eTK+teecLEy8QymZjjDTrCHg4x36
# 2AczdlQAIiq5TSAucGja5VP8g1zTnfL/RAxEZvLS471GABptArolXY2hMVHdVEYc
# TduZlu8aHARcphXveOB5/l3bPqpMVf2aFalv4ab733Aw6cPuQkbtwpMFifp9Y3s/
# 0HGBfADomK4OeDTDJfuvCp8ga907E48SjOJBGkh6c6B3ace2XH+CyB7+WBsoK6hs
# rV5twAXSe7frgP4lN/4Cm2isQl3D7vXM3PBQddI2aZzmewTfbgZptt4KCUhZh+t7
# FGB6ZKppQ++Rx0zsGN1s71MtjJnhXvJyPs9UyL1n7KQPTEX/07kwIwdMjxC/hpbZ
# mVq0mVccpMy7FYlTuiwFD+TEnhmxGDTVTJ267fcfrySVBHioA7vugeXaX3yLSqGQ
# dCWnsz5LyCxWvcfI7zjiXJLwefechLp0LWEBIH5+0fJPB1lfiy1DUutGDJTh9WZH
# eXfVVFsfrSQ3y0VaTqBESMjYsJnFFYQJ9tZJScBluOYacW6gqPGC6EU+bNYC1wpn
# gwVayaQQMYIDSTCCA0UCAQEwbzBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xv
# YmFsU2lnbiBudi1zYTExMC8GA1UEAxMoR2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcg
# Q0EgLSBTSEEzODQgLSBHNAIQARl1dHHJktdE36WW67lwFTALBglghkgBZQMEAgGg
# ggEtMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDArBgkqhkiG9w0BCTQxHjAc
# MAsGCWCGSAFlAwQCAaENBgkqhkiG9w0BAQsFADAvBgkqhkiG9w0BCQQxIgQgeEWK
# dvGFWM25V3KozZn9SmMTmoK6wSdZMKdqtJ5GHBMwgbAGCyqGSIb3DQEJEAIvMYGg
# MIGdMIGaMIGXBCALeaI5rkIQje9Ws1QFv4/NjlmnS4Tu4t7D2XHB6hc07DBzMF+k
# XTBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8G
# A1UEAxMoR2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEzODQgLSBHNAIQ
# ARl1dHHJktdE36WW67lwFTANBgkqhkiG9w0BAQsFAASCAYBKL6QwRc17Pje+Ai93
# 8lcuBL642LqyHB4uLrdzmhgvJuUsT+88TBynVrpkkxrZKChkh+P61JwruBVeoYJh
# eas5hiSwB0fCYj71T4DPfBJ+2FbL+Io78yCWiN7ZCyY43mRXASklBdeb35tmpzyD
# l7ewxPRJKSp36ku0Ree/B9Lw71VmpBJg+I2qvAfOFKknOhXp5T2vXXxhKA4yA4bA
# +Qq/Rkq9ycE7Xa1IDbXh5sFdCuonWZEjJ9ydnup5r/M4Ivg5vjLre6scqjgi6OWF
# rK1qTnMCIbbJb2zK+U2Wmrz0bKjIBRudgmZ2+b9zfCBmFWIa900ZbUuAbvWCpdqs
# v7Qp52GkLydJ+SePZlXGBehF7IbDeQpn6sgj9d2vzKCe8Vx3032uq4BtyHlbWBNh
# ccjKASZxr9o6WwIOgvD/bh8KSoE1sgHyZkJDk8v8JfrIh3wmWwTfDaYuwDkjuxVm
# RuMvclIMlT69ZGGwO59/ZODk/DTjvNPTd4euNw/jsZig4Xk=
# SIG # End signature block
